package de.viguard.ScreenKeyboard;

import javafx.beans.binding.Bindings;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.BooleanProperty;
import javafx.beans.property.ReadOnlyBooleanProperty;
import javafx.beans.property.ReadOnlyObjectProperty;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleStringProperty;
import javafx.beans.property.StringProperty;
import javafx.beans.value.ObservableStringValue;
import javafx.event.ActionEvent;
import javafx.event.EventHandler;
import javafx.event.EventTarget;
import javafx.event.EventType;
import javafx.geometry.Insets;
import javafx.geometry.Pos;
import javafx.scene.Node;
import javafx.scene.control.Button;
import javafx.scene.control.ContentDisplay;
import javafx.scene.control.ToggleButton;
import javafx.scene.input.KeyCode;
import javafx.scene.input.KeyEvent;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.HBox;
import javafx.scene.layout.Priority;
import javafx.scene.layout.VBox;
import javafx.scene.text.Font;
import javafx.scene.text.FontWeight;

public abstract class ScreenKeyboard {

	private static Font font = Font.font( "Courier New", FontWeight.BOLD, 24 );
	
	private Node root;

	public ScreenKeyboard( ReadOnlyObjectProperty< Node > target ) {}

	/**
	 * Creates a VirtualKeyboard which uses the focusProperty of the scene to which
	 * it is attached as its target
	 */
	public ScreenKeyboard() {
		this( null );
	}

	/**
	 * Visual component displaying this keyboard. The returned node has a style
	 * class of "virtual-keyboard". Buttons in the view have a style class of
	 * "virtual-keyboard-button".
	 * 
	 * @return a view of the keyboard.
	 */
	public Node view() {
		return root;
	}

	// Creates a "regular" button that has an unshifted and shifted value
	protected Button createShiftableButton( final String unshifted, final String shifted, final KeyCode code,
			Modifiers modifiers, final ReadOnlyObjectProperty< Node > target ) {
		final ReadOnlyBooleanProperty letter = new SimpleBooleanProperty(
				unshifted.length() == 1 && Character.isLetter( unshifted.charAt( 0 ) ) );
		final StringBinding text = Bindings.when( modifiers.shiftDown().or( modifiers.capsLockOn().and( letter ) ) )
				.then( shifted ).otherwise( unshifted );
		Button button = createButton( text, code, modifiers, target );
		button.setMaxSize( Double.MAX_VALUE, Double.MAX_VALUE );
		return button;
	}

	// Creates a button with fixed text not responding to Shift
	protected Button createNonshiftableButton( final String text, final KeyCode code, final Modifiers modifiers,
			final ReadOnlyObjectProperty< Node > target ) {
		StringProperty textProperty = new SimpleStringProperty( text );
		Button button = createButton( textProperty, code, modifiers, target );
		button.setMaxSize( Double.MAX_VALUE, Double.MAX_VALUE );
		return button;
	}

	// Creates a button with mutable text, and registers listener with it
	private Button createButton( final ObservableStringValue text, final KeyCode code, final Modifiers modifiers,
			final ReadOnlyObjectProperty< Node > target ) {
		final Button button = new Button();
		button.setFont( font );
		button.textProperty().bind( text );

		// Important not to grab the focus from the target:
		button.setFocusTraversable( false );

// Add a style class for css:
		button.getStyleClass().add( "virtual-keyboard-button" );

		button.setOnAction( new EventHandler< ActionEvent >() {
			@Override
			public void handle( ActionEvent event ) {

				final Node targetNode;
				if( target != null ) {
					targetNode = target.get();
				} else {
					targetNode = view().getScene().getFocusOwner();
				}

				if( targetNode != null ) {
					final String character;
					if( text.get().length() == 1 ) {
						character = text.get();
					} else {
						character = KeyEvent.CHAR_UNDEFINED;
					}
					final KeyEvent keyPressEvent = createKeyEvent( button, targetNode, KeyEvent.KEY_PRESSED, character,
							code, modifiers );
					targetNode.fireEvent( keyPressEvent );
					final KeyEvent keyReleasedEvent = createKeyEvent( button, targetNode, KeyEvent.KEY_RELEASED,
							character, code, modifiers );
					targetNode.fireEvent( keyReleasedEvent );
					if( character != KeyEvent.CHAR_UNDEFINED ) {
						final KeyEvent keyTypedEvent = createKeyEvent( button, targetNode, KeyEvent.KEY_TYPED,
								character, code, modifiers );
						targetNode.fireEvent( keyTypedEvent );
					}
					modifiers.releaseKeys();
				}
			}
		} );
		button.setMaxSize( Double.MAX_VALUE, Double.MAX_VALUE );
		button.setPrefSize( 80, 80 );
		return button;
	}

	// Utility method to create a KeyEvent from the Modifiers
	private KeyEvent createKeyEvent( Object source, EventTarget target, EventType< KeyEvent > eventType,
			String character, KeyCode code, Modifiers modifiers ) {
		return new KeyEvent( source, target, eventType, character, code.toString(), code, modifiers.shiftDown().get(),
				modifiers.ctrlDown().get(), modifiers.altDown().get(), modifiers.metaDown().get() );
	}

	// Utility method for creating cursor keys:
	private Button createCursorKey( KeyCode code, Modifiers modifiers, ReadOnlyObjectProperty< Node > target,
			Double... points ) {
		Button button = createNonshiftableButton( "", code, modifiers, target );
//    final Node graphic = PolygonBuilder.create().points(points).build();
//    graphic.setStyle("-fx-fill: -fx-mark-color;");
		// button.setGraphic(graphic);
		button.setContentDisplay( ContentDisplay.GRAPHIC_ONLY );
		button.setMaxSize( Double.MAX_VALUE, Double.MAX_VALUE );
		return button;
	}

	// Convenience class to bundle together the modifier keys and their selected
	// state
	protected static class Modifiers {
		private final ToggleButton shift;
		private final ToggleButton shift2;
		private final ToggleButton ctrl;
		private final ToggleButton alt;
		private final ToggleButton meta;
		private final ToggleButton capsLock;

		Modifiers() {
			this.shift = createToggle( "Shift" );
			this.shift2 = createToggle( "Shift" );
			this.ctrl = createToggle( "Ctrl" );
			this.alt = createToggle( "Alt" );
			this.meta = createToggle( "Meta" );
			this.capsLock = createToggle( "Caps" );

			shift2.selectedProperty().bindBidirectional( shift.selectedProperty() );
		}

		private ToggleButton createToggle( final String text ) {
			final ToggleButton tb = new ToggleButton( text );
			tb.setFont( font );
			tb.setFocusTraversable( false );
			tb.getStyleClass().add( "virtual-keyboard-button" );
			tb.setMaxSize( Double.MAX_VALUE, Double.MAX_VALUE );
			tb.setPrefSize( 80, 80 );
			return tb;
		}

		public Node shiftKey() {
			return shift;
		}

		public Node secondShiftKey() {
			return shift2;
		}

		public Node ctrlKey() {
			return ctrl;
		}

		public Node altKey() {
			return alt;
		}

		public Node metaKey() {
			return meta;
		}

		public Node capsLockKey() {
			return capsLock;
		}

		public BooleanProperty shiftDown() {
			return shift.selectedProperty();
		}

		public BooleanProperty ctrlDown() {
			return ctrl.selectedProperty();
		}

		public BooleanProperty altDown() {
			return alt.selectedProperty();
		}

		public BooleanProperty metaDown() {
			return meta.selectedProperty();
		}

		public BooleanProperty capsLockOn() {
			return capsLock.selectedProperty();
		}

		public void releaseKeys() {
			shift.setSelected( false );
			ctrl.setSelected( false );
			alt.setSelected( false );
			meta.setSelected( false );
		}
	}
}